/* ...............................................................

	WindowColors
	Copyright 1997-8 Steve Klingsporn <moofie@pobox.com>
	Based on WindowShade by Marco Nelissen <marcone@xs4all.nl>
	
		File:	WindowColors.cpp
	
	Contains:	Implementation of WindowColors API version 1.0b1.
	
	   Notes:	This is the documentation so far, sorry.
	   
   ............................................................... */

#ifndef _WINDOW_COLORS_H
#include "WindowColors.h"
#endif


/* ...............................................................
	Hack area id and name
	The global area_id and name for the hack area.
   ............................................................... */

const char	*APP_SERVER_AREA_NAME 			= "app_server_seg1";
const char	*APP_SERVER_NAME				= "app_server";
const char 	*HACK_AREA_NAME 	  			= "terrible_hack";
const char  *REQUIRED_SHORT_VERSION_STRING 	= "PR2";
area_id		_hackAreaID			 			= B_ERROR;
int8		_appServerCoolness				= -1;


/* ...............................................................
	_check_app_server_version()
	Returns B_OK if this app_server is okay to patch with
	WindowColors, and B_ERROR if not.  This function is called
	in every relevant WindowColors API routine for your
	protection.
   ............................................................... */

status_t check_app_server_version()
{
	status_t status = B_ERROR;	/* I'm too negative */
	
	switch (_appServerCoolness)
	{
		case -1:	/* Uninitialized */
		{
			BPath path; 
			
			if (find_directory(B_BEOS_SERVERS_DIRECTORY, &path) 
			    == B_NO_ERROR)
			{ 
				path.Append(APP_SERVER_NAME); 
				BFile app_server(path.Path(), O_RDONLY); 
			
				if (app_server.InitCheck() == B_NO_ERROR) 
				{ 
			    	BAppFileInfo info(&app_server); 
			
			    	if (info.InitCheck() == B_NO_ERROR) 
			    	{ 
						version_info version; 
				      	if ((info.GetVersionInfo(&version, 
				      							 B_APP_VERSION_KIND) 
				          	== B_NO_ERROR) && 
				          	strcmp(version.short_info, 
				          	REQUIRED_SHORT_VERSION_STRING) == 0)
			          	{
							status = B_OK;
							_appServerCoolness = 1;
						}
			    	} 
				} 
			}
		}
		break;
		
		case 1:	/* Cool */
			status = B_OK;
			break;
	}
	return status;
}


/* ...............................................................
	init_window_colors
	Initializes the WindowColors API.  Checks the version of the
	app_server first and returns an error if the version isn't
	supported by this version of WindowColors.  Otherwise, it
	sets up a clone hack of the "app_server_seg1" area, and if
	that works out okay, returns B_OK.
   ............................................................... */

status_t init_window_colors()
{
	status_t	status = B_ERROR;	/* Assume the worst */
	
	if (check_app_server_version() == B_OK &&
		_hackAreaID == B_ERROR)
	{
		_hackAreaID = find_area(HACK_AREA_NAME);
		status = B_OK;	/* Things should work from here... */
		
		/* Create our hack area if need be */
		if (_hackAreaID == B_NAME_NOT_FOUND)
		{
			/* Look for the app_server source area id */
			area_id		sourceAreaID = find_area(APP_SERVER_AREA_NAME);
			
			/* If the app_server's area ain't there, we're toast */
			if (sourceAreaID != B_NAME_NOT_FOUND)
			{
				char *address;
				
				/* Clone the area so we can mess with it */
				_hackAreaID = clone_area(HACK_AREA_NAME, &address,
										B_ANY_ADDRESS, B_READ_AREA |
										B_WRITE_AREA, sourceAreaID);
				/* Bail if something went wrong */
				if (_hackAreaID < B_OK)
					status = B_ERROR;
			}
			else
				status = B_ERROR;
		}
	}
	return status;
}


/* ...............................................................
	shut_down_window_colors()
	Cleans up by deleting the hack area we created.
   ............................................................... */

void shut_down_window_colors()
{
	/* Clean up the area we created, if need be */
	if (_hackAreaID != B_ERROR && _appServerCoolness == 1)
		delete_area(_hackAreaID);
}


/* ...............................................................
	compare_colors
	Returns true if the two colors are equal, and false if not.
	Ignores the value of the alpha component.
   ............................................................... */

bool compare_colors(rgb_color color1, rgb_color color2)
{
	return((color1.red == color2.red) &&
		   (color1.green == color2. green) &&
	       (color1.blue == color2.blue));
}


/* ...............................................................
	multiply_color
	Given an rgb_color and a float factor, return a a color with
	each component (r, g, b) multiplied by the factor, with the
	alpha component pinned at 255 for full coverage.
   ............................................................... */
  
rgb_color multiply_color(rgb_color color, float factor)
{
	color.red = min_c(255, color.red * factor);
	color.green = min_c(255, color.green * factor);
	color.blue = min_c(255, color.blue * factor);
	color.alpha = 255;	/* Full coverage */
	return color;
}


/* ...............................................................
	determine_hilite_color
	Given a window_color name and base rgb_color, return an
	appropriate hilite color for the given base color.
	Factors courtesy of Marco Nelissen (WindowShade).  These
	will be tweaked for 1.0.
   ............................................................... */

rgb_color determine_hilite_color(window_color which, rgb_color base)
{
	float factor;
	
	switch(which)
	{
		case ACTIVE_TAB_BASE_COLOR:
		case MINIMIZED_TAB_BASE_COLOR:
			factor = 1.33;
			break;
			
		case INACTIVE_TAB_BASE_COLOR:
		case INACTIVE_FRAME_BASE_COLOR:
		case INACTIVE_MODAL_FRAME_BASE_COLOR:
			factor = 1.09914;
			break;
			
		case ACTIVE_FRAME_BASE_COLOR:
		case ACTIVE_MODAL_FRAME_BASE_COLOR:
			factor = 1.18056;
			break;
			
		default:
			factor = 1.33;
			break;
	}
	return (multiply_color(base, factor));
}


/* ...............................................................
	determine_shadow_color
	Given a window_color name and base rgb_color, return an
	appropriate shadow color for the given base color.
   ............................................................... */

rgb_color determine_shadow_color(window_color which, rgb_color base)
{
	float factor;
	
	switch(which)
	{
		case ACTIVE_TAB_BASE_COLOR:
			factor = 0.75;
			break;
			
		case INACTIVE_TAB_BASE_COLOR:
		case INACTIVE_FRAME_BASE_COLOR:
		case INACTIVE_MODAL_FRAME_BASE_COLOR:
			factor = 0.931034;
			break;
			
		case MINIMIZED_TAB_BASE_COLOR:
			factor = 0.665845;
			break;
			
		case BORDER_COLOR:
			factor = 0.62;
			break;
			
		case ACTIVE_FRAME_BASE_COLOR:
		case ACTIVE_MODAL_FRAME_BASE_COLOR:
			factor = 0.851852;
			break;
			
		default:
			factor = 0.723447;
			break;
	}
	return (multiply_color(base, factor));
}


/* ...............................................................
	get_window_color
	Given a window_color name, returns the color for it in the
	app_server.
   ............................................................... */

rgb_color get_window_color(window_color which)
{
	rgb_color result = { 0, 0, 0, 255 };
	
	if (check_app_server_version() == B_OK)
	{
		area_info info;
		if (get_area_info(_hackAreaID, &info) == B_OK)
		{
			/* Snag a color from the app_server */
			char *address = (char *)info.address + 
							WINDOW_COLORS[which];
			::memcpy(&result, address, sizeof(rgb_color));
		}
	}
	return result;
}


/* ...............................................................
	get_window_colors
   ............................................................... */

void get_window_colors(rgb_color *first, set which)
{
	switch (which)
	{
		case ACTIVE_ONES:
		{
			for (uint8 i = 0; i < NUM_WINDOW_COLORS; i++)
			{
				*first = get_window_color((window_color)i);
				*first++;
			}
		}
		break;
		
		case DEFAULT_ONES:
		{
			for (uint8 i = 0; i < NUM_WINDOW_COLORS; i++)
			{
				*first = DEFAULT_WINDOW_COLORS[i];
				*first++;
			}
		}
	}
}


/* ...............................................................
	index_for_color
	A faster (?) version of BWindow::IndexForColor.  Given
	an rgb_color, it returns the index for the closest approximate
	color in the system color table.
   ............................................................... */

uint8 index_for_color(rgb_color color)
{
	return	(system_colors()->index_map[(((color.red & 0xf8) << 7)  |
										((color.green & 0xf8) << 2) |
										((color.blue & 0xf8) >> 3))]);
}


/* ...............................................................
	set_window_color
	Checks the version of the app_server, and if all is cool,
	sets the specified window_color to the color provided.  If
	refresh is true, the screen is refreshed as well.
   ............................................................... */

void set_window_color(window_color which,
					  rgb_color color, 
					  bool refresh)
{
	if (check_app_server_version() == B_OK)
	{
		area_info info;
		if (get_area_info(_hackAreaID, &info) == B_OK)
		{
			/* Lucifer & Aretha joint-venture... ;) */
			char *address = (char *)info.address + 
							WINDOW_COLORS[which];
			::memcpy(address, &color, sizeof(rgb_color));
							
			BBitmap *bitmap = new BBitmap(WINDOW_BITMAP_FRAME,
										  B_COLOR_8_BIT);
									
			bool getDefaults = false;	  
			if (which == ACTIVE_TAB_BASE_COLOR)
			{
				getDefaults = compare_colors(DEFAULT_WINDOW_COLORS
										    [which],
										    color);
				if (getDefaults)
					get_window_bitmap(bitmap, CLOSE_BOX_OFF_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(bitmap, 
										CLOSE_BOX_OFF_BITMAP, color);
				set_window_bitmap(bitmap, CLOSE_BOX_OFF_BITMAP);
				
				if (getDefaults)
					get_window_bitmap(bitmap, CLOSE_BOX_ON_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(bitmap, 
										CLOSE_BOX_ON_BITMAP, color);
										
				set_window_bitmap(bitmap, CLOSE_BOX_ON_BITMAP);

				if (getDefaults)
					get_window_bitmap(bitmap, ZOOM_BOX_OFF_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(bitmap, 
										ZOOM_BOX_OFF_BITMAP, color);
				set_window_bitmap(bitmap, ZOOM_BOX_OFF_BITMAP);

				if (getDefaults)
					get_window_bitmap(bitmap, ZOOM_BOX_ON_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(bitmap, 
										ZOOM_BOX_ON_BITMAP, color);
				set_window_bitmap(bitmap, ZOOM_BOX_ON_BITMAP);
			}
			
			if (which == MINIMIZED_TAB_BASE_COLOR)
			{
				getDefaults = compare_colors(DEFAULT_WINDOW_COLORS
							    [which],
							    color);
				if (getDefaults)
					get_window_bitmap(bitmap,
									  MINIMIZED_CLOSE_BOX_OFF_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(bitmap,
										MINIMIZED_CLOSE_BOX_OFF_BITMAP,
										color);
				set_window_bitmap(bitmap, MINIMIZED_CLOSE_BOX_OFF_BITMAP);
				
				if (getDefaults)
					get_window_bitmap(bitmap,
									  MINIMIZED_CLOSE_BOX_ON_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(bitmap,
										MINIMIZED_CLOSE_BOX_ON_BITMAP,
										color);
				set_window_bitmap(bitmap, MINIMIZED_CLOSE_BOX_ON_BITMAP);
			}
			
			if (which == ACTIVE_FRAME_BASE_COLOR)
			{
				getDefaults = compare_colors(DEFAULT_WINDOW_COLORS
											 [which], color);
				
				if (getDefaults)
					get_window_bitmap(bitmap, RESIZE_CORNER_BITMAP,
									  DEFAULT_ONES);
				else
					color_window_bitmap(bitmap, RESIZE_CORNER_BITMAP,
										color);
				set_window_bitmap(bitmap, RESIZE_CORNER_BITMAP);
			}
			
			delete bitmap;
			/* Clean up the screen */
			if (refresh)
				refresh_windows();
			
			/* Tell everyone in town about it... */
			be_roster->Broadcast(&BMessage(WINDOW_COLORS_CHANGED));
		}
	}
}


/* ...............................................................
	set_window_colors
   ............................................................... */

void set_window_colors(rgb_color *first, bool refresh)
{
	for (uint8 which = 0; which < NUM_WINDOW_COLORS; which++)
	{
		set_window_color((window_color)which, *first,
						 (refresh &&
						   which == NUM_WINDOW_COLORS - 1) );
		first++;
	}
}


/* ...............................................................
	refresh_windows
	Refreshes all of the windows on the screen so they show
	their new, true colors.
   ............................................................... */

void refresh_windows()
{
	BScreen screen;
	BRect frame = screen.Frame();
	frame.InsetBy(-2, -2);
	
	/* Create a window the size of the screen and then some */
	BWindow	*window = new BWindow(frame, "", B_BORDERED_WINDOW, 0);
	window->Show();
	
	/* Lock it and blow it away */
	if (window->Lock())
		window->Quit();
}


/* ...............................................................
	get_window_bitmap
	Given a BBitmap pointer, it sets the bits in the bitmap to
	the specified window bitmap.
   ............................................................... */

void get_window_bitmap(BBitmap *bitmap, window_bitmap which,
					   set ones)
{	
	if (bitmap != NULL && check_app_server_version() == B_OK)
	{
		area_info info;
		uchar *address;
		if (get_area_info(_hackAreaID, &info) == B_OK)
		{
			if (ones == ACTIVE_ONES)
				address = (uchar *)info.address +
							 	 window_bitmaps[which];
			else
				address = (uchar *)DEFAULT_WINDOW_BITMAPS[which];
			bitmap->SetBits(address, 224, 0, B_COLOR_8_BIT);
		}
	}
}


/* ...............................................................
	set_window_bitmap
	Sets the window_bitmap in the app_server specified with the
	bitmap data from the BBitmap passed in.
   ............................................................... */

void set_window_bitmap(BBitmap *bitmap, window_bitmap which)
{
	if (bitmap != NULL && check_app_server_version() == B_OK)
	{
		area_info info;
		if (get_area_info(_hackAreaID, &info) == B_OK)
		{
			uchar *address = (uchar *)info.address +
							 window_bitmaps[which];
			::memcpy(address, bitmap->Bits(), 224);
		}
	}
}


/* ...............................................................
	color_window_bitmap
	Sets the window_bitmap in the app_server specified with the
	bitmap data from the BBitmap passed in.
	
	I want to mess with these values for 1.0.  I'm not happy with
	how it colors bitmaps.  It's probably also not cool for Intel.
   ............................................................... */
   
void color_window_bitmap(BBitmap *bitmap, window_bitmap which,
						 rgb_color base_color)
{
	if (bitmap != NULL)
	{
		uchar *original = (uchar *)DEFAULT_WINDOW_BITMAPS[which];
		uchar *address = (uchar *)bitmap->Bits();
		for (uint8 i = 0; i < 224; i++)
		{
			float factor;
			switch (*original)
			{
				case 0x98:
				case 0x5d:
				case 0x94:
					factor = 0.80;
					break;
					
				case 0xe5:
				case 0x3f:
				case 0x40:
					factor = 1.15;
					break;
					
				case 0xf9:
				case 0x1b:
				case 0x67:
					factor = 1.0;
					break;
					
				case 0x0c:
					factor = 0.45;
					break;
					
				case 0xbd:
				case 0x6d:
					factor = 0.91;
					break;
					
				case 0xfc:
					factor = 1.33;
					break;
					
			}
			rgb_color color = system_colors()->color_list[*original];
			color = multiply_color(base_color, factor);
			*address = index_for_color(color);
			original++;
			address++;
		}
	}
}